<?php

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use App\Http\Controllers\API\PropertyController;
use App\Http\Controllers\API\LeadController;
use App\Http\Controllers\API\AgentController;
use App\Http\Controllers\API\BlogController;
use App\Http\Controllers\API\AuthController;
use App\Http\Controllers\Admin\DashboardController;
use App\Http\Controllers\Admin\PropertyController as AdminPropertyController;
use App\Http\Controllers\Admin\LeadController as AdminLeadController;
use App\Http\Controllers\Admin\AgentController as AdminAgentController;
use App\Http\Controllers\Admin\SettingsController;

/*
|--------------------------------------------------------------------------
| API Routes - Renuma Real Estate
|--------------------------------------------------------------------------
*/

// =============================================
// PUBLIC API ROUTES
// =============================================

Route::prefix('v1')->group(function () {
    
    // Properties
    Route::prefix('properties')->group(function () {
        Route::get('/', [PropertyController::class, 'index']);
        Route::get('/featured', [PropertyController::class, 'featured']);
        Route::get('/latest', [PropertyController::class, 'latest']);
        Route::get('/search', [PropertyController::class, 'search']);
        Route::get('/statistics', [PropertyController::class, 'statistics']);
        Route::get('/price-range', [PropertyController::class, 'priceRange']);
        Route::get('/{slug}', [PropertyController::class, 'show']);
        Route::get('/{id}/similar', [PropertyController::class, 'similar']);
    });

    // Agents
    Route::prefix('agents')->group(function () {
        Route::get('/', [AgentController::class, 'index']);
        Route::get('/featured', [AgentController::class, 'featured']);
        Route::get('/{id}', [AgentController::class, 'show']);
        Route::get('/{id}/properties', [PropertyController::class, 'byAgent']);
    });

    // Leads & Inquiries
    Route::prefix('leads')->group(function () {
        Route::post('/', [LeadController::class, 'store']);
        Route::post('/viewing', [LeadController::class, 'requestViewing']);
        Route::post('/callback', [LeadController::class, 'requestCallback']);
    });

    // Blog
    Route::prefix('blog')->group(function () {
        Route::get('/', [BlogController::class, 'index']);
        Route::get('/featured', [BlogController::class, 'featured']);
        Route::get('/categories', [BlogController::class, 'categories']);
        Route::get('/{slug}', [BlogController::class, 'show']);
        Route::get('/category/{slug}', [BlogController::class, 'byCategory']);
    });

    // Locations
    Route::get('/locations', function () {
        return \App\Models\Location::active()->get();
    });

    // Property Types
    Route::get('/property-types', function () {
        return \App\Models\PropertyType::active()->get();
    });

    // Features
    Route::get('/features', function () {
        return \App\Models\Feature::all()->groupBy('category');
    });

    // Testimonials
    Route::get('/testimonials', function () {
        return \App\Models\Testimonial::where('status', 'active')
            ->orderBy('sort_order')
            ->get();
    });

    // Settings
    Route::get('/settings', function () {
        $settings = \App\Models\Setting::all()->pluck('value', 'key');
        return response()->json(['success' => true, 'data' => $settings]);
    });

    // Contact Form
    Route::post('/contact', [LeadController::class, 'store']);

    // Pages
    Route::get('/pages/{slug}', function ($slug) {
        $page = \App\Models\Page::where('slug', $slug)
            ->where('status', 'active')
            ->firstOrFail();
        return response()->json(['success' => true, 'data' => $page]);
    });
});

// =============================================
// ADMIN API ROUTES (Protected)
// =============================================

Route::prefix('admin')->group(function () {
    
    // Authentication
    Route::post('/login', [AuthController::class, 'login']);
    Route::post('/register', [AuthController::class, 'register']);
    
    // Protected routes
    Route::middleware(['auth:sanctum', 'admin'])->group(function () {
        
        // Auth
        Route::post('/logout', [AuthController::class, 'logout']);
        Route::get('/me', [AuthController::class, 'me']);
        
        // Dashboard
        Route::get('/dashboard', [DashboardController::class, 'index']);
        Route::get('/dashboard/analytics', [DashboardController::class, 'analytics']);
        Route::get('/dashboard/export', [DashboardController::class, 'export']);
        
        // Properties Management
        Route::prefix('properties')->group(function () {
            Route::get('/', [AdminPropertyController::class, 'index']);
            Route::post('/', [AdminPropertyController::class, 'store']);
            Route::get('/{id}', [AdminPropertyController::class, 'show']);
            Route::put('/{id}', [AdminPropertyController::class, 'update']);
            Route::delete('/{id}', [AdminPropertyController::class, 'destroy']);
            Route::post('/{id}/toggle-featured', [AdminPropertyController::class, 'toggleFeatured']);
            Route::post('/bulk-action', [AdminPropertyController::class, 'bulkAction']);
            Route::get('/form/options', [AdminPropertyController::class, 'formOptions']);
            
            // Property Images
            Route::post('/{id}/images', [AdminPropertyController::class, 'uploadImages']);
            Route::delete('/images/{imageId}', [AdminPropertyController::class, 'deleteImage']);
        });
        
        // Agents Management
        Route::apiResource('agents', AdminAgentController::class);
        Route::post('agents/{id}/toggle-featured', [AdminAgentController::class, 'toggleFeatured']);
        
        // Leads Management
        Route::prefix('leads')->group(function () {
            Route::get('/', [AdminLeadController::class, 'index']);
            Route::get('/{id}', [AdminLeadController::class, 'show']);
            Route::put('/{id}', [AdminLeadController::class, 'update']);
            Route::delete('/{id}', [AdminLeadController::class, 'destroy']);
            Route::post('/{id}/assign', [AdminLeadController::class, 'assign']);
            Route::post('/{id}/status', [AdminLeadController::class, 'updateStatus']);
            Route::post('/{id}/notes', [AdminLeadController::class, 'addNote']);
            Route::get('/export/csv', [AdminLeadController::class, 'exportCsv']);
        });
        
        // Blog Management
        Route::apiResource('blog', \App\Http\Controllers\Admin\BlogController::class);
        Route::apiResource('blog-categories', \App\Http\Controllers\Admin\BlogCategoryController::class);
        
        // Testimonials
        Route::apiResource('testimonials', \App\Http\Controllers\Admin\TestimonialController::class);
        Route::post('testimonials/{id}/reorder', [\App\Http\Controllers\Admin\TestimonialController::class, 'reorder']);
        
        // Pages CMS
        Route::apiResource('pages', \App\Http\Controllers\Admin\PageController::class);
        
        // Users Management
        Route::apiResource('users', \App\Http\Controllers\Admin\UserController::class);
        
        // Settings
        Route::get('/settings', [SettingsController::class, 'index']);
        Route::put('/settings', [SettingsController::class, 'update']);
        Route::post('/settings/logo', [SettingsController::class, 'uploadLogo']);
        Route::post('/settings/favicon', [SettingsController::class, 'uploadFavicon']);
        
        // Email Templates
        Route::apiResource('email-templates', \App\Http\Controllers\Admin\EmailTemplateController::class);
        
        // Activity Logs
        Route::get('/activity-logs', function () {
            return \App\Models\ActivityLog::with('user')->latest()->paginate(50);
        });
        
        // Reports
        Route::get('/reports/properties', [\App\Http\Controllers\Admin\ReportController::class, 'properties']);
        Route::get('/reports/leads', [\App\Http\Controllers\Admin\ReportController::class, 'leads']);
        Route::get('/reports/revenue', [\App\Http\Controllers\Admin\ReportController::class, 'revenue']);
    });
});

// =============================================
// HEALTH CHECK
// =============================================

Route::get('/health', function () {
    return response()->json([
        'status' => 'ok',
        'app' => 'Renuma Real Estate API',
        'version' => '1.0.0',
        'timestamp' => now()->toDateTimeString()
    ]);
});
