<?php
/**
 * Renuma Real Estate - Installation Processor
 */

header('Content-Type: application/json');
session_start();

$response = ['success' => false, 'message' => ''];

try {
    // Get request data
    $input = json_decode(file_get_contents('php://input'), true);
    $action = $input['action'] ?? '';
    $database = $input['database'] ?? $_SESSION['db_config'] ?? [];
    $config = $input['config'] ?? [];
    
    // Base paths
    $backendPath = realpath(__DIR__ . '/../backend');
    $envPath = $backendPath . '/.env';
    $envExamplePath = $backendPath . '/.env.example';
    
    switch ($action) {
        case 'create_env':
            createEnvFile($envPath, $envExamplePath, $database, $config);
            $response['success'] = true;
            $response['message'] = 'Environment file created';
            break;
            
        case 'generate_key':
            $key = generateAppKey();
            updateEnvFile($envPath, 'APP_KEY', $key);
            $response['success'] = true;
            $response['message'] = 'Application key generated';
            break;
            
        case 'setup_database':
            setupDatabase($database);
            $response['success'] = true;
            $response['message'] = 'Database connection verified';
            break;
            
        case 'run_migrations':
            runMigrations($backendPath);
            $response['success'] = true;
            $response['message'] = 'Database tables created';
            break;
            
        case 'seed_data':
            seedDatabase($backendPath);
            $response['success'] = true;
            $response['message'] = 'Sample data installed';
            break;
            
        case 'storage_link':
            createStorageLink($backendPath);
            $response['success'] = true;
            $response['message'] = 'Storage symlink created';
            break;
            
        case 'optimize':
            optimizeApplication($backendPath);
            $response['success'] = true;
            $response['message'] = 'Application optimized';
            break;
            
        case 'set_permissions':
            setPermissions($backendPath);
            $response['success'] = true;
            $response['message'] = 'File permissions set';
            break;
            
        case 'finalize':
            finalizeInstallation($backendPath, $config);
            $response['success'] = true;
            $response['message'] = 'Installation finalized';
            break;
            
        default:
            throw new Exception('Invalid action');
    }
    
} catch (Exception $e) {
    $response['message'] = $e->getMessage();
}

echo json_encode($response);

// Helper Functions

function createEnvFile($envPath, $envExamplePath, $database, $config) {
    if (!file_exists($envExamplePath)) {
        throw new Exception('.env.example file not found');
    }
    
    $envContent = file_get_contents($envExamplePath);
    
    // Update database credentials
    $envContent = preg_replace('/DB_HOST=.*/', 'DB_HOST=' . $database['db_host'], $envContent);
    $envContent = preg_replace('/DB_DATABASE=.*/', 'DB_DATABASE=' . $database['db_name'], $envContent);
    $envContent = preg_replace('/DB_USERNAME=.*/', 'DB_USERNAME=' . $database['db_user'], $envContent);
    $envContent = preg_replace('/DB_PASSWORD=.*/', 'DB_PASSWORD=' . $database['db_password'], $envContent);
    
    // Update site configuration
    $envContent = preg_replace('/APP_URL=.*/', 'APP_URL=' . $config['app_url'], $envContent);
    $envContent = preg_replace('/APP_ENV=.*/', 'APP_ENV=production', $envContent);
    $envContent = preg_replace('/APP_DEBUG=.*/', 'APP_DEBUG=false', $envContent);
    
    if (!empty($config['whatsapp'])) {
        $envContent .= "\nWHATSAPP_NUMBER=" . $config['whatsapp'];
    }
    
    if (!file_put_contents($envPath, $envContent)) {
        throw new Exception('Could not create .env file');
    }
}

function generateAppKey() {
    return 'base64:' . base64_encode(random_bytes(32));
}

function updateEnvFile($envPath, $key, $value) {
    $envContent = file_get_contents($envPath);
    
    if (preg_match("/^$key=/m", $envContent)) {
        $envContent = preg_replace("/^$key=.*/m", "$key=$value", $envContent);
    } else {
        $envContent .= "\n$key=$value";
    }
    
    file_put_contents($envPath, $envContent);
}

function setupDatabase($database) {
    $dsn = "mysql:host={$database['db_host']};dbname={$database['db_name']};charset=utf8mb4";
    $pdo = new PDO($dsn, $database['db_user'], $database['db_password']);
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
}

function runMigrations($backendPath) {
    chdir($backendPath);
    
    // Read and execute SQL file
    $sqlFile = dirname(__DIR__) . '/database-schema.sql';
    
    if (!file_exists($sqlFile)) {
        throw new Exception('Database schema file not found');
    }
    
    $sql = file_get_contents($sqlFile);
    
    // Get database connection from .env
    require $backendPath . '/vendor/autoload.php';
    $dotenv = Dotenv\Dotenv::createImmutable($backendPath);
    $dotenv->load();
    
    $pdo = new PDO(
        "mysql:host=" . $_ENV['DB_HOST'] . ";dbname=" . $_ENV['DB_DATABASE'] . ";charset=utf8mb4",
        $_ENV['DB_USERNAME'],
        $_ENV['DB_PASSWORD']
    );
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    
    // Split SQL into individual statements
    $statements = array_filter(
        array_map('trim', explode(';', $sql)),
        function($statement) {
            return !empty($statement) && 
                   strpos($statement, '--') !== 0 &&
                   strpos($statement, '/*') !== 0;
        }
    );
    
    foreach ($statements as $statement) {
        if (!empty($statement)) {
            try {
                $pdo->exec($statement);
            } catch (PDOException $e) {
                // Ignore table already exists errors
                if (strpos($e->getMessage(), 'already exists') === false) {
                    throw $e;
                }
            }
        }
    }
}

function seedDatabase($backendPath) {
    chdir($backendPath);
    
    $output = shell_exec('php artisan db:seed --class=DummyDataSeeder --force 2>&1');
    
    if (strpos($output, 'error') !== false || strpos($output, 'Error') !== false) {
        throw new Exception('Database seeding failed: ' . $output);
    }
}

function createStorageLink($backendPath) {
    chdir($backendPath);
    
    $publicStorage = $backendPath . '/public/storage';
    $storageApp = $backendPath . '/storage/app/public';
    
    // Remove existing link if it exists
    if (file_exists($publicStorage)) {
        if (is_link($publicStorage)) {
            unlink($publicStorage);
        }
    }
    
    // Create symlink
    if (!symlink($storageApp, $publicStorage)) {
        throw new Exception('Could not create storage symlink');
    }
}

function optimizeApplication($backendPath) {
    chdir($backendPath);
    
    $commands = [
        'php artisan config:cache',
        'php artisan route:cache',
        'php artisan view:cache'
    ];
    
    foreach ($commands as $command) {
        shell_exec($command . ' 2>&1');
    }
}

function setPermissions($backendPath) {
    // Set directory permissions
    $directories = [
        $backendPath . '/storage',
        $backendPath . '/bootstrap/cache'
    ];
    
    foreach ($directories as $dir) {
        if (file_exists($dir)) {
            chmod($dir, 0775);
            
            // Recursively set permissions
            $iterator = new RecursiveIteratorIterator(
                new RecursiveDirectoryIterator($dir, RecursiveDirectoryIterator::SKIP_DOTS),
                RecursiveIteratorIterator::SELF_FIRST
            );
            
            foreach ($iterator as $item) {
                chmod($item->getPathname(), $item->isDir() ? 0775 : 0664);
            }
        }
    }
}

function finalizeInstallation($backendPath, $config) {
    // Create a file to indicate installation is complete
    $installFile = $backendPath . '/.installed';
    file_put_contents($installFile, date('Y-m-d H:i:s'));
    
    // Clear any existing caches
    chdir($backendPath);
    shell_exec('php artisan cache:clear 2>&1');
    
    // Create installation log
    $logContent = "Installation completed on " . date('Y-m-d H:i:s') . "\n";
    $logContent .= "Site URL: " . $config['app_url'] . "\n";
    $logContent .= "Admin Email: " . $config['admin_email'] . "\n";
    
    file_put_contents($backendPath . '/storage/logs/installation.log', $logContent);
}
