<?php

namespace App\Http\Controllers\API;

use App\Http\Controllers\Controller;
use App\Models\Property;
use App\Models\PropertyType;
use App\Models\Location;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\DB;

class PropertyController extends Controller
{
    /**
     * Display a listing of properties with filters
     */
    public function index(Request $request)
    {
        try {
            $query = Property::with(['propertyType', 'location', 'agent', 'images'])
                ->published()
                ->available();

            // Search
            if ($request->has('search')) {
                $query->search($request->search);
            }

            // Filters
            $query->filterByType($request->type)
                  ->filterByLocation($request->location)
                  ->filterByPrice($request->min_price, $request->max_price)
                  ->filterByBedrooms($request->bedrooms)
                  ->filterByBathrooms($request->bathrooms);

            // Purpose filter (sale/rent)
            if ($request->has('purpose')) {
                if ($request->purpose === 'sale') {
                    $query->forSale();
                } elseif ($request->purpose === 'rent') {
                    $query->forRent();
                }
            }

            // Featured only
            if ($request->has('featured') && $request->featured) {
                $query->featured();
            }

            // Sorting
            $sortBy = $request->get('sort_by', 'created_at');
            $sortOrder = $request->get('sort_order', 'desc');
            
            $allowedSorts = ['price', 'created_at', 'views', 'title'];
            if (in_array($sortBy, $allowedSorts)) {
                $query->orderBy($sortBy, $sortOrder);
            }

            // Pagination
            $perPage = $request->get('per_page', 12);
            $properties = $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => $properties,
                'filters' => [
                    'types' => PropertyType::active()->get(),
                    'locations' => Location::active()->get(),
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching properties',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get featured properties for homepage
     */
    public function featured()
    {
        try {
            $featured = Cache::remember('featured_properties', 3600, function () {
                return Property::with(['propertyType', 'location', 'agent', 'images'])
                    ->published()
                    ->available()
                    ->featured()
                    ->orderBy('created_at', 'desc')
                    ->limit(6)
                    ->get();
            });

            return response()->json([
                'success' => true,
                'data' => $featured
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching featured properties'
            ], 500);
        }
    }

    /**
     * Get latest properties
     */
    public function latest()
    {
        try {
            $latest = Property::with(['propertyType', 'location', 'agent', 'images'])
                ->published()
                ->available()
                ->orderBy('created_at', 'desc')
                ->limit(8)
                ->get();

            return response()->json([
                'success' => true,
                'data' => $latest
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching latest properties'
            ], 500);
        }
    }

    /**
     * Get similar properties
     */
    public function similar($id)
    {
        try {
            $property = Property::findOrFail($id);
            
            $similar = Property::with(['propertyType', 'location', 'images'])
                ->where('id', '!=', $id)
                ->published()
                ->available()
                ->where(function ($query) use ($property) {
                    $query->where('property_type_id', $property->property_type_id)
                          ->orWhere('location_id', $property->location_id);
                })
                ->whereBetween('price', [
                    $property->price * 0.7,
                    $property->price * 1.3
                ])
                ->limit(4)
                ->get();

            return response()->json([
                'success' => true,
                'data' => $similar
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching similar properties'
            ], 500);
        }
    }

    /**
     * Display the specified property
     */
    public function show($slug)
    {
        try {
            $property = Property::with([
                    'propertyType',
                    'location.parent',
                    'agent',
                    'images',
                    'features'
                ])
                ->where('slug', $slug)
                ->published()
                ->firstOrFail();

            // Increment views
            $property->incrementViews();

            return response()->json([
                'success' => true,
                'data' => $property
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Property not found'
            ], 404);
        }
    }

    /**
     * Get property statistics
     */
    public function statistics()
    {
        try {
            $stats = Cache::remember('property_stats', 1800, function () {
                return [
                    'total_properties' => Property::published()->count(),
                    'for_sale' => Property::published()->forSale()->count(),
                    'for_rent' => Property::published()->forRent()->count(),
                    'sold' => Property::where('status', 'sold')->count(),
                    'total_value' => Property::published()->available()->sum('price'),
                    'average_price' => Property::published()->available()->avg('price'),
                    'by_type' => PropertyType::withCount(['properties' => function ($query) {
                        $query->published()->available();
                    }])->get(),
                    'by_location' => Location::withCount(['properties' => function ($query) {
                        $query->published()->available();
                    }])->orderBy('properties_count', 'desc')->limit(10)->get(),
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $stats
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching statistics'
            ], 500);
        }
    }

    /**
     * Search properties with advanced filters
     */
    public function search(Request $request)
    {
        try {
            $query = Property::with(['propertyType', 'location', 'agent', 'images'])
                ->published()
                ->available();

            // Text search
            if ($request->filled('q')) {
                $query->search($request->q);
            }

            // Price range
            if ($request->filled('min_price')) {
                $query->where('price', '>=', $request->min_price);
            }
            if ($request->filled('max_price')) {
                $query->where('price', '<=', $request->max_price);
            }

            // Area range
            if ($request->filled('min_area')) {
                $query->where('area_sqft', '>=', $request->min_area);
            }
            if ($request->filled('max_area')) {
                $query->where('area_sqft', '<=', $request->max_area);
            }

            // Bedrooms & Bathrooms
            if ($request->filled('bedrooms')) {
                $query->where('bedrooms', '>=', $request->bedrooms);
            }
            if ($request->filled('bathrooms')) {
                $query->where('bathrooms', '>=', $request->bathrooms);
            }

            // Property type
            if ($request->filled('type')) {
                $query->where('property_type_id', $request->type);
            }

            // Location
            if ($request->filled('location')) {
                $query->where('location_id', $request->location);
            }

            // Purpose
            if ($request->filled('purpose')) {
                if ($request->purpose === 'sale') {
                    $query->forSale();
                } else if ($request->purpose === 'rent') {
                    $query->forRent();
                }
            }

            // Features
            if ($request->filled('features')) {
                $featureIds = is_array($request->features) 
                    ? $request->features 
                    : explode(',', $request->features);
                
                $query->whereHas('features', function ($q) use ($featureIds) {
                    $q->whereIn('features.id', $featureIds);
                });
            }

            // Sort
            $sortBy = $request->get('sort', 'created_at');
            $sortOrder = $request->get('order', 'desc');
            $query->orderBy($sortBy, $sortOrder);

            // Pagination
            $perPage = min($request->get('per_page', 12), 50);
            $results = $query->paginate($perPage);

            return response()->json([
                'success' => true,
                'data' => $results
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Search error',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get properties by agent
     */
    public function byAgent($agentId)
    {
        try {
            $properties = Property::with(['propertyType', 'location', 'images'])
                ->where('agent_id', $agentId)
                ->published()
                ->available()
                ->orderBy('created_at', 'desc')
                ->paginate(12);

            return response()->json([
                'success' => true,
                'data' => $properties
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching agent properties'
            ], 500);
        }
    }

    /**
     * Get property price range
     */
    public function priceRange()
    {
        try {
            $range = Cache::remember('property_price_range', 3600, function () {
                return [
                    'min' => Property::published()->available()->min('price'),
                    'max' => Property::published()->available()->max('price'),
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $range
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error fetching price range'
            ], 500);
        }
    }
}
