<?php
/**
 * Renuma Real Estate - Database Connection Tester
 */

header('Content-Type: application/json');

$response = ['success' => false, 'message' => ''];

try {
    // Get POST data
    $host = $_POST['db_host'] ?? 'localhost';
    $name = $_POST['db_name'] ?? '';
    $user = $_POST['db_user'] ?? '';
    $pass = $_POST['db_password'] ?? '';
    
    // Validate inputs
    if (empty($name) || empty($user)) {
        throw new Exception('Database name and username are required');
    }
    
    // Try to connect
    $dsn = "mysql:host=$host;dbname=$name;charset=utf8mb4";
    $options = [
        PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION,
        PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
        PDO::ATTR_EMULATE_PREPARES => false,
    ];
    
    $pdo = new PDO($dsn, $user, $pass, $options);
    
    // Test if we can query
    $stmt = $pdo->query('SELECT VERSION() as version');
    $version = $stmt->fetch();
    
    // Check MySQL version
    if (version_compare($version['version'], '5.7.0', '<')) {
        $response['message'] = 'MySQL version is ' . $version['version'] . '. Version 5.7 or higher is recommended.';
        $response['success'] = true;
        $response['warning'] = true;
    } else {
        $response['success'] = true;
        $response['message'] = 'Connection successful! MySQL version: ' . $version['version'];
    }
    
    // Save credentials to session for installation
    session_start();
    $_SESSION['db_config'] = [
        'host' => $host,
        'name' => $name,
        'user' => $user,
        'pass' => $pass
    ];
    
} catch (PDOException $e) {
    $response['message'] = 'Database connection failed: ' . $e->getMessage();
    
    // Provide helpful error messages
    if (strpos($e->getMessage(), 'Access denied') !== false) {
        $response['message'] .= '<br><br><strong>Tip:</strong> Check your username and password.';
    } elseif (strpos($e->getMessage(), 'Unknown database') !== false) {
        $response['message'] .= '<br><br><strong>Tip:</strong> Make sure the database exists and includes your cPanel username prefix.';
    } elseif (strpos($e->getMessage(), "Can't connect") !== false) {
        $response['message'] .= '<br><br><strong>Tip:</strong> Check if MySQL is running and the host is correct (usually "localhost").';
    }
    
} catch (Exception $e) {
    $response['message'] = $e->getMessage();
}

echo json_encode($response);
